/*
 * @(#) $Id: timestamps.c 342 2023-10-06 23:39:02Z leres $ (XSE)
 *
 * Copyright (c) 1999, 2000, 2001, 2003, 2008, 2009, 2010, 2013, 2015, 2018, 2023
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#define DIRENTRY struct dirent
#endif
#ifdef HAVE_SYS_DIR_H
#define DIRENTRY struct direct
#include <sys/dir.h>
#endif

#ifndef HAVE_STRLCAT
#include "strlcat.h"
#endif

/* Globals */
const char *prog;
int debug;
int errors;
int iflag;
int lflag;
int sflag;

extern char *optarg;
extern int optind, opterr, optopt;

/* Forwards */
int invisible(const char *);
int main(int, char **);
int process(const char *, int);
void usage(void) __attribute__((noreturn));
int mysort(const void *, const void *);

int
main(int argc, char **argv)
{
	int i, op;
	char *cp;

	if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	while ((op = getopt(argc, argv, "ils")) != EOF)
		switch (op) {

		case 'i':
			++iflag;
			break;

		case 'l':
			++lflag;
			break;

		case 's':
			++sflag;
			break;

		default:
			usage();
		}

	if (argc == optind)
		(void)process(".", 1);
	else for (i = optind; i < argc; ++i)
		(void)process(argv[i], 1);

	exit(errors != 0);
}

int
process(const char *name, int dirok)
{
	char *cp, **cpp, **names;
	const char *p;
	int i, n, size;
	DIR *dirp;
	DIRENTRY *dp;
	struct timespec *mtimp;
	struct stat sbuf;
	int nfiles;
	char temp[2048];

	if (stat(name, &sbuf) < 0) {
		fprintf(stderr, "%s: process: stat %s: %s\n",
		    prog, name, strerror(errno));
		return (0);
	}

	if (!S_ISDIR(sbuf.st_mode)) {
		p = name;
		if (sflag) {
			cp = strrchr(p, '/');
			if (cp != NULL)
				p = cp + 1;
		}
		if (iflag)
			mtimp = &sbuf.st_ctim;
		else
			mtimp = &sbuf.st_mtim;
		printf("%s\t%ld", p, mtimp->tv_sec);
		if (lflag)
			printf(".%09ld", mtimp->tv_nsec);
		putchar('\n');
		return (1);
	} if (!dirok)
		return (0);

	nfiles = 0;
	dirp = opendir(name);
	if (dirp == NULL) {
		fprintf(stderr, "%s: process: opendir %s: %s\n",
		    prog, name, strerror(errno));
		exit(1);
	}
	size = 32;
	names = (char **)malloc(size * sizeof(*names));
	if (names == NULL) {
		fprintf(stderr, "%s: process: malloc: %s\n",
		    prog, strerror(errno));
		exit(1);
	}
	cpp = names;
	n = 0;
	while ((dp = readdir(dirp)) != NULL) {
		if (*dp->d_name == '.')
			continue;
		if (invisible(dp->d_name))
			continue;
		strcpy(temp, name);
		if (strcmp(name, ".") == 0 || strcmp(name, "./") == 0)
			cp = dp->d_name;
		else {
			strcpy(temp, name);
			cp = temp + strlen(temp) - 1;
			if (cp >= temp && *cp != '/')
				strlcat(temp, "/", sizeof(temp));
			strlcat(temp, dp->d_name, sizeof(temp));
			cp = temp;
		}

		if (n >= size) {
			i = cpp - names;
			size <<= 2;
			names = (char **)realloc(names, size * sizeof(*names));
			if (names == NULL) {
				fprintf(stderr, "%s: process: realloc: %s\n",
				    prog, strerror(errno));
				exit(1);
			}
			cpp = names + i;
		}
		*cpp = strdup(cp);
		if (*cpp == NULL) {
			fprintf(stderr, "%s: process: strdup: %s\n",
			    prog, strerror(errno));
			exit(1);
		}
		++cpp;
		++n;
	}
	(void)closedir(dirp);

	qsort(names, n, sizeof(*names), mysort);

	for (i = 0, cpp = names; i < n; ++i, ++cpp) {
		nfiles += process(*cpp, 0);
		free(*cpp);
	}
	free(names);
	return (nfiles);
}

int
invisible(const char *fn)
{
	const char *p;

	if (strcmp(fn, "..") == 0)
		return (0);
	if (*fn == '.')
		return (1);
	p = fn + strlen(fn) - 1;
	if (p >= fn && *p == '-')
		return (1);
	return (0);
}

int
mysort(const void *a1, const void *a2)
{
	const char *s1, *s2;

	s1 = *(const char **)a1;
	s2 = *(const char **)a2;
	return (strcoll(s1, s2));
}

void
usage(void)
{

	fprintf(stderr, "usage: %s [-ils] [file ...]\n", prog);
	fprintf(stderr, "    -i inode time\n");
	fprintf(stderr, "    -l nanoseconds\n");
	fprintf(stderr, "    -s simple filename\n");
	exit(1);
}
