#!/bin/sh
# @(#) $Id: pngrotate.sh 369 2025-11-07 23:23:49Z leres $ (XSE)
#
# pngrotate - rotate png images
#

prog="`basename $0`"
usage()
{
	echo "${prog} -l file ...	# rotate left" 1>&2
	echo "${prog} -r file ...	# rotate right" 1>&2
	echo "${prog} -f file ...	# rotate 180 (flip)" 1>&2
	echo "${prog} -h file ...	# horizontal" 1>&2
	echo "${prog} -V file ...	# vertical" 1>&2
	echo "${prog} -u file ...	# undo" 1>&2
	echo "${prog} -n		# dryrun" 1>&2
	echo "${prog} -F		# force" 1>&2
	echo "${prog} -O		# override" 1>&2
	exit 1
}

PATH=/usr/local/bin:${PATH}
export PATH

dryrun=0
force=0
override=0
status=0
undo=0

while getopts flnruFO opt ; do
	case "$opt" in

	f)
		flags="-filter:v transpose=clock,transpose=clock"
		;;

	l)
		flags="-filter:v transpose=cclock"
		;;

	n)
		dryrun=1
		;;

	r)
		flags="-filter:v transpose=clock"
		;;

	u)
		undo=1
		;;

	F)
		force=1
		;;

	O)
		override=1
		;;

	?)
		usage
		;;
	esac
done

shift $((${OPTIND} - 1))

if [ -z "${flags}" -a ${undo} -eq 0 ]; then
	usage
fi

flags="-loglevel error ${flags}"

if [ "${force}" -ne 0 -a ${override} -ne 0 ]; then
	echo "${prog}: Cannot use both -F (force) and -O (override)" 1>&2
	exit 1
fi

if [ $# -lt 1 ]; then
	usage
fi

# Execute (or just echo if -n)
execute() {
	cmd="$*"
	if [ ${dryrun} -eq 0 ]; then
		eval "${cmd}"
		return $?
	fi
	echo "+ ${cmd}"
	return 0
}

if [ ${undo} -ne 0 ]; then
	# Undo logic
	while [ $# -gt 0 ]; do
		r="$1"
		shift

		f="`echo ${r} |
		    sed -e 's/r\.png$/.png/' -e 's/r\.PNG$/.PNG/'`"
		if [ "${f}" = "${r}" ]; then
			echo "${prog}: ${r} already unrotated? (skipping)"
			status=1
			continue
		fi
		if [ -r ${r} -a -r ${f} ]; then
			echo "${prog}: Both ${r} and ${f} exist (skipping)"
			status=1
			continue
		fi
		if [ -r ${f} ]; then
			echo "${prog}: ${f} already exists (skipping)"
			status=1
			continue
		fi
		if [ ! -r ${r} ]; then
			echo "${prog}: Can't find ${r}"
			status=1
			continue
		fi
		o="${f}-"
		if [ ! -r ${o} ]; then
			echo "${prog}: ${o} missing (skipping)"
			status=1
			continue
		fi

		execute "rm -f ${r}-"
		execute "mv ${r} ${r}-"
		execute "mv ${o} ${f}"
		echo "${prog}: ${r} -> ${f}"
	done
	exit ${status}
fi

# Make sure we have everything we need
for f in ffmpeg; do
	which ${f} > /dev/null 2>&1
	if [ $? -ne 0 ]; then
		echo "${prog}: Can't find ${f}" 1>&2
		exit 1
	fi
done

# Rotate logic
while [ $# -gt 0 ]; do
	f="$1"
	shift

	case "`echo ${f} | tr A-Z a-z`" in

	*.png)
		# Ok
		;;

	*)
		echo "${prog}: ${f} unexpected file extension (skipping)"
		status=1
		continue
		;;
	esac

	r="`echo ${f} | sed -e 's/r*\(\.[Pp][Nn][Gg]\)$/r\1/'`"
	if [ "${f}" = "${r}" -a ${override} -eq 0 ]; then
		echo "${prog}: ${f} already rotated? (use -O to override)"
		status=1
		continue
	fi
	if [ -r ${r} -a ${override} -eq 0 ]; then
		echo "${prog}: ${r} already exists, skipping ${f}"
		status=1
		continue
	fi
	if [ ! -r ${f} ]; then
		echo "${prog}: Can't find ${f}"
		status=1
		continue
	fi
	o="${f}-"
	if [ -r ${o} -a ${force} -eq 0 ]; then
		echo "${prog}: ${o} already exists (skipping)"
		status=1
		continue
	fi

	# Transform
	t="${r%%.png}.$$.png"
	execute "ffmpeg -i ${f} ${flags} ${t}"
	if [ $? -ne 0 ]; then
		rm ${t}
		exit 1
	fi

	# Do this last in case ffmpeg fails
	if [ ${override} -eq 0 ]; then
		# Don't keep original when overriding
		execute "mv ${f} ${o}"
	else
		# Pick up timestamps from original
		o="${o%r\.png-}.png-"
	fi

	# Rename temporary then copy access and modified timestamps
	execute "mv ${t} ${r}"
	execute "touch -r ${o} ${r}"
	if [ ${override} -eq 0 ]; then
		echo "${prog}: ${f} -> ${r}"
	else
		echo "${prog}: ${r} (rotated)"
	fi
done

exit ${status}
