/*
 * @(#) $Id: jpeg.c 358 2025-07-07 00:29:27Z leres $ (XSE)
 *
 * Copyright (c) 1999, 2000, 2008, 2009, 2010, 2015, 2017, 2023, 2024, 2025
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>

#include "exif.h"
#include "jpeg.h"

/* Define a few JPEG markers */
#define M_SOF0 0xc0
#define M_SOF3 0xc3
#define M_SOI 0xd8
#define M_EOI 0xd9
#define M_SOS 0xda

/* Forwards */
static int get(FILE *, u_char *);
static int jpeg(FILE *, int *, int *);
static int mustbe(FILE *, u_char);

int
parsejpeg(const char *image, int *wp, int *hp, struct timespec *mtimp,
    char *location, size_t locationsize)
{
	int status;
	FILE *f;
	u_char ch;
	struct stat sbuf;

	f = fopen(image, "r");
	if (f == NULL)
		return (0);

	/* XXX ignore errors */
	if (mtimp != NULL && fstat(fileno(f), &sbuf) >= 0)
		*mtimp = sbuf.st_mtim;

	if (!get(f, &ch)) {
		fclose(f);
		return (0);
	}
	if (ch == 0xff)
		status = jpeg(f, wp, hp);
	else
		status = 0;

	if (location != NULL && locationsize > 0) {
		location[0] = '\0';
		exiflocation(f, location, locationsize);
	}
	fclose(f);
	return (status);
}

static int
get(FILE * f, u_char *chP)
{
	int ich;

	ich = getc(f);
	if (ich == EOF)
		return (0);
	*chP = (u_char) ich;
	return (1);
}

static int
jpeg(FILE *f, int *wp, int *hp)
{
	int l, w, h;
	int i;
	u_char ch, l1, l2, w1, w2, h1, h2;

	/* Read rest of initial marker. */
	if (!mustbe(f, M_SOI))
		return (0);

	/*
	 * JPEG blocks consist of a 0xff, a marker byte, a block size
	 * (two bytes, big-endian), and the rest of the block. The
	 * block size includes itself - i.e. is the block size was 2
	 * then there would be no additional bytes in the block.
	 *
	 * So, what we do here is read blocks until we get an SOF
	 * marker, and then extract the width and height from that.
	 */
	for (;;) {
		if (!mustbe(f, 0xff))
			return (0);
		if (!get(f, &ch))
			return (0);
		if (!get(f, &l1))
			return (0);
		if (!get(f, &l2))
			return (0);
		l = (int) l1 *256 + (int) l2;
		/* Is it the block we're looking for? */
		if (ch >= M_SOF0 && ch <= M_SOF3) {
			/* Toss the sample precision. */
			if (!get(f, &ch))
				return (0);
			/* Read the height and width. */
			if (!get(f, &h1))
				return (0);
			if (!get(f, &h2))
				return (0);
			if (!get(f, &w1))
				return (0);
			if (!get(f, &w2))
				return (0);
			w = (int) w1 *256 + (int) w2;
			h = (int) h1 *256 + (int) h2;
			*wp = w;
			*hp = h;
			return (1);
		}
		if ( ch == M_SOS || ch == M_EOI )
			return (0);
		/* Not the block we want.  Read and toss. */
		for (i = 2; i < l; ++i)
			if (!get(f, &ch))
				return (0);
	}
}

static int
mustbe(FILE * f, u_char ch)
{
	u_char ch2;

	if (!get(f, &ch2))
		return (0);
	if (ch2 != ch)
		return (0);
	return (1);
}
